'use strict';

const { parentPort, workerData } = require('worker_threads');
const fs = require('fs');
const path = require('path');

// Import comparison functions
const sharp = require('sharp');

const RESIZE_WIDTH = 32;
const RESIZE_HEIGHT = 32;

async function loadImageRawRgba(imagePath) {
  return await sharp(imagePath).ensureAlpha().resize(RESIZE_WIDTH, RESIZE_HEIGHT).raw().toBuffer();
}

function computeDirectSimilarityFromBuf(buf1, buf2) {
  const minLen = Math.min(buf1.length, buf2.length);
  let diff = 0;
  for (let i = 0; i < minLen; i += 4) {
    diff += Math.abs(buf1[i] - buf2[i]);
    diff += Math.abs(buf1[i + 1] - buf2[i + 1]);
    diff += Math.abs(buf1[i + 2] - buf2[i + 2]);
  }
  const maxDiff = (minLen / 4) * 255 * 3;
  return maxDiff === 0 ? 1 : 1 - (diff / maxDiff);
}

async function compareWithMSE(templatePath, framePath) {
  try {
    const buf1 = await loadImageRawRgba(templatePath);
    const buf2 = await loadImageRawRgba(framePath);
    const similarity = computeDirectSimilarityFromBuf(buf1, buf2);
    return { similarity, method: 'mse' };
  } catch (error) {
    return { similarity: 0, error: error.message };
  }
}

async function compareWithPixelmatch(templatePath, framePath) {
  try {
    const pixelmatch = require('pixelmatch');
    const buf1 = await loadImageRawRgba(templatePath);
    const buf2 = await loadImageRawRgba(framePath);
    const numPixels = RESIZE_WIDTH * RESIZE_HEIGHT;
    const diff = Buffer.alloc(numPixels * 4);
    const numDiff = pixelmatch(buf1, buf2, diff, RESIZE_WIDTH, RESIZE_HEIGHT, { threshold: 0.1, alpha: 0.1 });
    const similarity = Math.max(0, 1 - (numDiff / numPixels));
    return { similarity, numDiff };
  } catch (error) {
    return { similarity: 0, numDiff: RESIZE_WIDTH * RESIZE_HEIGHT };
  }
}

async function compareWithOpenCV(templatePath, framePath) {
  try {
    const buf1 = await loadImageRawRgba(templatePath);
    const buf2 = await loadImageRawRgba(framePath);
    const similarity = computeDirectSimilarityFromBuf(buf1, buf2);
    return { similarity, method: 'opencv' };
  } catch (error) {
    return { similarity: 0, error: error.message };
  }
}

async function compareWithPHash(templatePath, framePath) {
  try {
    const templateBuf = await sharp(templatePath).resize(32, 32).grayscale().raw().toBuffer();
    const frameBuf = await sharp(framePath).resize(32, 32).grayscale().raw().toBuffer();

    let sum1 = 0, sum2 = 0;
    for (let i = 0; i < templateBuf.length; i++) {
      sum1 += templateBuf[i];
      sum2 += frameBuf[i];
    }
    const avg1 = sum1 / templateBuf.length;
    const avg2 = sum2 / frameBuf.length;

    let distance = 0;
    for (let i = 0; i < templateBuf.length; i++) {
      const b1 = templateBuf[i] >= avg1 ? 1 : 0;
      const b2 = frameBuf[i] >= avg2 ? 1 : 0;
      if (b1 !== b2) distance++;
    }

    const similarity = Math.max(0, 1 - (distance / 1024));
    return { similarity, method: 'phash', distance };
  } catch (error) {
    return { similarity: 0, error: error.message };
  }
}

async function compareWithSSIM(templatePath, framePath) {
  try {
    const ssim = require('ssim.js').default;
    const templateImg = await sharp(templatePath).resize(64, 64).raw().toBuffer({ resolveWithObject: true });
    const frameImg = await sharp(framePath).resize(64, 64).raw().toBuffer({ resolveWithObject: true });

    const templateData = new Uint8ClampedArray(templateImg.data);
    const frameData = new Uint8ClampedArray(frameImg.data);

    const templateImageData = { data: templateData, width: 64, height: 64 };
    const frameImageData = { data: frameData, width: 64, height: 64 };

    const result = ssim(templateImageData, frameImageData, { k1: 0.01, k2: 0.03, bitDepth: 8 });
    return { similarity: result.mssim, method: 'ssim' };
  } catch (error) {
    return { similarity: 0, error: error.message };
  }
}

async function compareWithMultiScale(templatePath, framePath) {
  try {
    const scales = [16, 32, 64];
    let totalScore = 0;

    for (const size of scales) {
      const templateBuf = await sharp(templatePath).resize(size, size).raw().toBuffer();
      const frameBuf = await sharp(framePath).resize(size, size).raw().toBuffer();

      let diff = 0;
      const minLen = Math.min(templateBuf.length, frameBuf.length);
      for (let i = 0; i < minLen; i++) {
        diff += Math.abs(templateBuf[i] - frameBuf[i]);
      }
      const pixelScore = 1 - (diff / (minLen * 255));
      totalScore += pixelScore;
    }

    const finalScore = totalScore / scales.length;
    return { similarity: Math.max(0, Math.min(1, finalScore)), method: 'multiscale' };
  } catch (error) {
    return { similarity: 0, error: error.message };
  }
}

async function compareRegions(templatePath, framePath, method) {
  switch (method) {
    case 'fast':
      return await compareWithMSE(templatePath, framePath);
    case 'balanced':
      return await compareWithPixelmatch(templatePath, framePath);
    case 'accurate':
      return await compareWithOpenCV(templatePath, framePath);
    case 'phash':
      return await compareWithPHash(templatePath, framePath);
    case 'ssim':
      return await compareWithSSIM(templatePath, framePath);
    case 'multiscale':
      return await compareWithMultiScale(templatePath, framePath);
    default:
      return await compareWithMSE(templatePath, framePath);
  }
}

// Process messages from main thread
parentPort.on('message', async (task) => {
  const { templatePath, framePaths, method, threshold } = task;

  const results = [];
  for (const frame of framePaths) {
    try {
      const result = await compareRegions(templatePath, frame.path, method);
      let thumbnailBase64 = null;
      try {
        thumbnailBase64 = fs.readFileSync(frame.path).toString('base64');
      } catch (e) {}
      results.push({
        frame,
        result,
        thumbnailBase64
      });
    } catch (err) {
      results.push({ frame, result: { similarity: 0 }, error: err.message });
    }
  }

  parentPort.postMessage({ results });
});
